<?php
// Assumes $conn and $website are already defined earlier in your application.
// -------------------------------------------------------------------------
// Defensive initializations (preserve your original logic; just make safe)


require_once "../core/db-connect91.php";

$selected_staff_id = (int)($_SESSION['selected_staff_id'] ?? $selected_staff_id ?? 0);
$staff_id = (int)($_SESSION['selected_staff_id'] ?? 0);


// NOTE: you previously had $month = 6 and $year = 2024 in your code.
// I preserved those values to avoid changing your logic.
$month = $_POST['month'] ?? 0;
$year = $_POST['year'] ?? 0;

// Batch fallback
$batch = $_POST['batch'] ?? ($batch ?? '');


?>




<?php
// -------------------------------------------------------------------------
// 3) Salary & attendance calculations
// 3.1 Net salary (type = 'net_salary')
$net_sql = "SELECT amount FROM staff_salary_components WHERE staff_id = ? AND type = 'net_salary' LIMIT 1";
$net_stmt = $conn->prepare($net_sql);
$net_stmt->bind_param("i", $staff_id);
$net_stmt->execute();
$net_res = $net_stmt->get_result();
$net_salary = ($net_res && $net_res->num_rows > 0) ? (float)$net_res->fetch_assoc()['amount'] : 0.0;
$net_stmt->close();

// 3.2 Attendance counts
$att_sql = "
SELECT status, leave_type, COUNT(*) as count
FROM staff_attendance
WHERE staff_id = ? AND month = ? AND year = ?
GROUP BY status, leave_type
";
$att_stmt = $conn->prepare($att_sql);
$att_stmt->bind_param("iii", $staff_id, $month, $year);
$att_stmt->execute();
$att_res = $att_stmt->get_result();

$present = $absent = $half_day = $leave = $paid_leave = 0;
while ($r = $att_res->fetch_assoc()) {
    switch ($r['status']) {
        case 'present':
            $present += (int)$r['count'];
            break;
        case 'absent':
            $absent += (int)$r['count'];
            break;
        case 'half_day':
            $half_day += (int)$r['count'];
            break;
        case 'leave':
            $leave += (int)$r['count'];
            if (strtolower($r['leave_type']) === 'paid') {
                $paid_leave += (int)$r['count'];
            }
            break;
    }
}
$att_stmt->close();

// 3.3 Total days & daily salary
$total_days = $present + $absent + $half_day + $leave;
$daily_salary = ($total_days > 0) ? ($net_salary / $total_days) : 0.0;

// 3.4 Payable total (present + half_day/2 + paid_leave)
$payable_total = ($present * $daily_salary) + ($half_day * $daily_salary * 0.5) + ($paid_leave * $daily_salary);

// 3.5 Deductions (sum where type='deduction')
$deduction_sql = "SELECT COALESCE(SUM(amount),0) as total_deduction FROM staff_salary_components WHERE staff_id = ? AND type = 'deduction'";
$ded_stmt = $conn->prepare($deduction_sql);
$ded_stmt->bind_param("i", $staff_id);
$ded_stmt->execute();
$ded_res = $ded_stmt->get_result();
$total_deduction = ($ded_res && $ded_res->num_rows > 0) ? (float)$ded_res->fetch_assoc()['total_deduction'] : 0.0;
$ded_stmt->close();

// 3.6 Net payable after deductions
$net_payable = max(0.0, $payable_total - $total_deduction);

// 3.7 Sum of payments already made for the same month/year/batch
$payment_sum_sql = "SELECT COALESCE(SUM(payment_amount), 0) AS total_payment
        FROM staff_salary_payment_history
        WHERE staff_id = ? AND salary_month = ? AND salary_year = ? AND batch = ?";
$payment_sum_stmt = $conn->prepare($payment_sum_sql);
$payment_sum_stmt->bind_param("iiis", $staff_id, $month, $year, $batch);
$payment_sum_stmt->execute();
$payment_sum_res = $payment_sum_stmt->get_result();
$total_payment = 0.0;
if ($payment_sum_res && $row = $payment_sum_res->fetch_assoc()) {
    $total_payment = (float)$row['total_payment'];
}
$payment_sum_stmt->close();

// 3.8 Final net payable considering payments done (keeps your original variable)
$final_net_payable = $net_payable - $total_payment;
?>

<div class="card mt-4">
    <div class="p-2 text-lg font-bold">Salary & Attendance Summary</div>
    <div class="px-4 pb-5 sm:px-1 w-full">
        <table class="table-auto w-full border border-slate-300 text-lg font-bold">
            <tbody>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Net Salary (₹)</td>
                    <td class="px-3 py-2 text-right"><?= number_format($net_salary,2) ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Total Days</td>
                    <td class="px-3 py-2 text-right"><?= $total_days ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Present Days</td>
                    <td class="px-3 py-2 text-right"><?= $present ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Absent Days</td>
                    <td class="px-3 py-2 text-right"><?= $absent ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Half Days</td>
                    <td class="px-3 py-2 text-right"><?= $half_day ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Leave Days</td>
                    <td class="px-3 py-2 text-right"><?= $leave ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Paid Leave Days</td>
                    <td class="px-3 py-2 text-right"><?= $paid_leave ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Payable Total (₹)</td>
                    <td class="px-3 py-2 text-right"><?= number_format($payable_total,2) ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Deductions (₹)</td>
                    <td class="px-3 py-2 text-right"><?= number_format($total_deduction,2) ?></td>
                </tr>
                <tr class="border-b border-slate-300">
                    <td class="px-3 py-2">Net Payable (₹)</td>
                    <td class="px-3 py-2 text-right"><?= number_format($net_payable,2) ?></td>
                </tr>
                <tr class="border-b border-slate-300">
	                <td class="px-3 py-2">Payment Done (₹)</td>
	                <td class="px-3 py-2 text-right"><?= number_format($total_payment,2) ?></td>
                </tr>
				<tr class="border-b border-slate-300">
				    <td class="px-3 py-2">Amount To Pay (₹)</td>
				    <td class="px-3 py-2 text-right"><?= number_format($final_net_payable,2) ?></td>
				</tr>
            </tbody>
        </table>
    </div>
</div>

<?php
// -------------------------------------------------------------------------
// 4) Staff bank details
$bank_sql = "SELECT b.*, s.staff_name, s.staff_role 
        FROM staff_bank_details b
        JOIN staff s ON b.staff_id = s.staff_id
        WHERE b.staff_id = ?
        ORDER BY b.id ASC";
$bank_stmt = $conn->prepare($bank_sql);
$bank_stmt->bind_param("i", $selected_staff_id);
$bank_stmt->execute();
$bank_result = $bank_stmt->get_result();
?>

<div class="mt-5">
    <div class="is-scrollbar-hidden min-w-full overflow-x-auto">
        <table class="is-zebra w-full text-left">
            <thead>
                <tr>
                    <th class="whitespace-nowrap rounded-l-lg bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">#</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">Name</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">Role</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">Bank</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">Account Number</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">IFSC</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">UPI ID</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">Payment Number</th>
                    <th class="whitespace-nowrap bg-slate-200 px-4 py-3 font-semibold uppercase text-slate-800 lg:px-5">PAN</th>
                    <th class="whitespace-nowrap rounded-r-lg bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Aadhaar</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $count = 1;
                if ($bank_result->num_rows > 0) {
                    while ($row = $bank_result->fetch_assoc()) {
                        echo '<tr>';
                        echo '<td class="whitespace-nowrap rounded-l-lg px-4 py-3 sm:px-5">' . $count++ . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['staff_name']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['staff_role']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['bank_name']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['account_number']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['ifsc_code']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['upi_id']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['payment_number']) . '</td>';
                        echo '<td class="whitespace-nowrap px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['pan_number']) . '</td>';
                        echo '<td class="whitespace-nowrap rounded-r-lg px-4 py-3 sm:px-5" style="cursor:pointer;color:blue;" title="Click to copy" onclick="copyCell(this)">' . htmlspecialchars($row['aadhaar_number']) . '</td>';
                        echo '</tr>';
                    }
                } else {
                    echo '<tr><td colspan="10" class="text-center px-4 py-3">No staff bank details found</td></tr>';
                }
                ?>
            </tbody>
        </table>
    </div>
</div>

<?php
$bank_stmt->close();
?>

<script>
function copyCell(cell) {
    let text = cell.textContent.trim();
    navigator.clipboard.writeText(text).then(() => {
        alert("Copied: " + text);
    }).catch(err => {
        alert("Failed to copy!");
        console.error(err);
    });
}
</script>

<?php
// -------------------------------------------------------------------------
// 5) Payment History table (preserve your SELECT fields & ordering)
$history_sql = "SELECT 
            p.id,
            CONCAT(s.staff_name, ' ', s.staff_surname) AS staff_name,
            s.staff_role,
            p.total_amount,
            p.deductions,
            p.net_payable,
            p.payment_amount,
            p.due_amount,            
            p.payment_status,
            p.payment_date,
            p.payment_method,
            p.payment_reference,
            p.remarks,
            p.present_days,
            p.absent_days,
            p.half_days,
            p.paid_leaves
        FROM staff_salary_payment_history p
        INNER JOIN staff s ON s.staff_id = p.staff_id
        WHERE p.staff_id = ?
          AND p.salary_month = ?
          AND p.salary_year = ?
          AND p.batch = ?
        ORDER BY p.payment_date DESC";
$history_stmt = $conn->prepare($history_sql);
$history_stmt->bind_param("iiis", $staff_id, $month, $year, $batch);
$history_stmt->execute();
$history_result = $history_stmt->get_result();
?>

<div class="mt-5">
  <div class="is-scrollbar-hidden min-w-full overflow-x-auto">
    <table class="is-zebra w-full text-left">
      <thead>
        <tr>
          <th class="whitespace-nowrap rounded-l-lg bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">#</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Name</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Role</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Total Salary</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Deductions</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Net Payable</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Paid</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Due</th>        
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Status</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Payment Date</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Method</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Reference</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Present</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Absent</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Half Days</th>
          <th class="whitespace-nowrap  bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Paid Leaves</th>
          <th class="whitespace-nowrap rounded-r-lg bg-slate-200 px-3 py-3 font-semibold uppercase text-slate-800 lg:px-5">Remarks</th>
        </tr>
      </thead>
      <tbody>
        <?php
        if ($history_result->num_rows > 0) {
            $i = 1;
            $total_paid = 0;
            while ($row = $history_result->fetch_assoc()) {
                $total_paid += (float)$row['payment_amount'];
                echo "<tr>
                        <td class='whitespace-nowrap rounded-l-lg px-4 py-3 sm:px-5'>{$i}</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['staff_name'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['staff_role'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>₹".htmlspecialchars($row['total_amount'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>₹".htmlspecialchars($row['deductions'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>₹".htmlspecialchars($row['net_payable'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>₹".htmlspecialchars($row['payment_amount'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>₹".htmlspecialchars($row['due_amount'])."</td>                      
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['payment_status'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['payment_date'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['payment_method'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['payment_reference'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['present_days'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['absent_days'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['half_days'])."</td>
                        <td class='whitespace-nowrap  px-4 py-3 sm:px-5'>".htmlspecialchars($row['paid_leaves'])."</td>
                        <td class='whitespace-nowrap rounded-r-lg px-4 py-3 sm:px-5'>".htmlspecialchars($row['remarks'])."</td>
                      </tr>";
                $i++;
            }
            echo "<tr class='bg-slate-100 font-semibold'>
                    <td colspan='6' class='text-right'>Total Paid:</td>
                    <td class='text-green-600'>₹".number_format($total_paid,2)."</td>
                    <td colspan='9'></td>
                  </tr>";
        } else {
            echo "<tr><td colspan='16' class='text-center'>No payment history found</td></tr>";
        }
        ?>
      </tbody>
    </table>
  </div>
</div>

<?php
$history_stmt->close();
?>

<!-- Payment form -->
<form method="POST">
<div class="card mt-4">
<div class="p-2 text-right"></div>
<div class="flex grow flex-col items-center px-4 pb-5 sm:px-1 w-full">
<label class="relative flex">
    <span class="font-bold text-lg">Amount To Pay (₹)</span>   
    <input 
        name="payable_amount"
        class="form-input peer w-full rounded-lg border border-slate-300 bg-transparent px-3 py-2 pl-9 
               placeholder:text-slate-400/70 hover:border-slate-400 focus:border-primary 
               dark:border-navy-450 dark:hover:border-navy-400 dark:focus:border-accent 
               font-bold text-lg"
        type="text"
        value="<?php echo number_format($final_net_payable, 2, '.', '') ?>"
    />
</label>
</div>
</div>

<!-- Payment Method -->
<div class="card mt-4">
    <div class="p-2 text-right"></div>
    <div class="flex grow flex-col items-center px-4 pb-5 sm:px-1 w-full">
        <label class="block w-full">
            <span>Payment Method</span>
            <select id="paymentMethod" name="payment_method" required
                class="mt-1.5 w-full rounded-lg border border-slate-300 bg-transparent px-3 py-2
                       hover:border-slate-400 focus:border-primary
                       dark:border-navy-450 dark:bg-navy-700
                       dark:hover:border-navy-400 dark:focus:border-accent">
                <option value="">Select Payment Method</option>
                <option value="cash">Cash</option>
                <option value="cheque">Cheque</option>
                <option value="upi">UPI</option>
                <option value="imps">IMPS</option>
                <option value="neft">NEFT</option>
                <option value="rtgs">RTGS</option>
                <option value="demand_draft">Demand Draft</option>
                <option value="others">Others</option>
            </select>
        </label>
    </div>
</div>

<div id="paymentDetailsContainer" class="card mt-4 hidden">
    <div class="p-4 space-y-4" id="paymentDetails"></div>
</div><!-- Dynamic Payment Details -->

<script>
const methodSelect = document.getElementById("paymentMethod");
const detailsContainer = document.getElementById("paymentDetailsContainer");
const detailsBox = document.getElementById("paymentDetails");

methodSelect.addEventListener("change", function() {
    const method = this.value;
    detailsBox.innerHTML = ""; 
    detailsContainer.classList.add("hidden");
    if (!method) return;

    let fields = "";

    switch (method) {

        case "cheque":
            fields = `
                <h3 class="font-bold">Cheque Payment Details</h3>
                <label class="block">
                    <span>Cheque Number</span>
                    <input type="text" name="cheque_number" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>Bank Name</span>
                    <input type="text" name="bank_name" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>IFSC Code</span>
                    <input type="text" name="ifsc_code" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
            `;
            break;

        case "upi":
            fields = `
                <h3 class="font-bold">UPI Payment Details</h3>
                <label class="block">
                    <span>UPI ID</span>
                    <input type="text" name="upi_id" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>Transaction ID</span>
                    <input type="text" name="transaction_id" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
            `;
            break;

        case "imps":
        case "neft":
        case "rtgs":
            fields = `
                <h3 class="font-bold">${method.toUpperCase()} Payment Details</h3>
                <label class="block">
                    <span>Bank Name</span>
                    <input type="text" name="bank_name" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>IFSC Code</span>
                    <input type="text" name="ifsc_code" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>Transaction ID</span>
                    <input type="text" name="transaction_id" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
            `;
            break;

        case "demand_draft":
            fields = `
                <h3 class="font-bold">Demand Draft Payment Details</h3>
                <label class="block">
                    <span>DD Number</span>
                    <input type="text" name="dd_number" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
                <label class="block">
                    <span>Bank Name</span>
                    <input type="text" name="bank_name" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
            `;
            break;

        case "others":
            fields = `
                <h3 class="font-bold">Other Payment Method</h3>
                <label class="block">
                    <span>Description</span>
                    <input type="text" name="other_description" class="mt-1.5 w-full rounded-lg border px-3 py-2" required />
                </label>
            `;
            break;
    }

    detailsBox.innerHTML = fields;
    detailsContainer.classList.remove("hidden");
});

// Live cash calculation
detailsBox.addEventListener("input", function(e) {
    if (methodSelect.value === "cash") {
        let total = 0;
        document.querySelectorAll(".note-count").forEach(input => {
            let count = parseInt(input.value) || 0;
            let value = parseInt(input.dataset.noteValue) || 0;
            if (isNaN(value) && input.dataset.noteValue === "Other") {
                total += count; // Other amount directly
            } else {
                total += count * value;
            }
        });
        document.getElementById("cashTotal").textContent = total;
    }
});
</script>
    <input type="hidden" name="form_type" value="staff_salary_payment">
<div class="mt-6 grid w-full grid-cols-1 gap-2">                
    <button type="submit" 
        class="btn space-x-2 bg-primary px-0 font-medium text-white hover:bg-primary-focus focus:bg-primary-focus active:bg-primary-focus/90 dark:bg-accent dark:hover:bg-accent-focus dark:focus:bg-accent-focus dark:active:bg-accent/90"
      >
Submit
    </button>
</div>
</form>

<?php
// -------------------------------------------------------------------------
// 6) Handle POST (insert/update) - keep your insert/update behavior and fields
if (
    $_SERVER['REQUEST_METHOD'] === 'POST' &&
    isset($_POST['form_type']) &&
    $_POST['form_type'] === 'staff_salary_payment'
) {
    // Your existing payment processing code here

// Escape / cast inputs
$staff_id       = (int)$_SESSION['selected_staff_id'];
$batch          = mysqli_real_escape_string($conn, $_POST['batch'] ?? $batch);
$salary_month   = (int)$month;
$salary_year    = (int)$year;
$total_amount   = (float)$payable_total;
$deductions     = (float)$total_deduction;
$net_payable    = (float)$net_payable;
$payment_amount = (float)($_POST['payable_amount'] ?? 0.0);
// Define due_amount
$due_amount = max(0.0, $net_payable - $payment_amount);

// validation

$errors = [];

// Payment amount can't be negative
if ($payment_amount < 0) {
    $errors[] = "Payment amount cannot be negative.";
}

// Payment amount can't exceed net payable
if ($payment_amount > $net_payable) {
    $errors[] = "Payment amount cannot exceed net payable.";
}

// Calculate due amount
$due_amount = $net_payable - $payment_amount;

// Due amount can't be negative (should already be handled above, but double-check)
if ($due_amount < 0) {
    $errors[] = "Due amount cannot be negative.";
}

// If there are validation errors, stop here
if (!empty($errors)) {
    foreach ($errors as $error) {
        echo "<div style='color:red;'>Error: $error</div>";
    }
    exit;
}

// Set payment status based on due amount
if ($due_amount == 0) {
$payment_status = "paid";
} elseif ($due_amount > 0 && $due_amount < $net_payable) {
$payment_status = "partial";
} else {
$payment_status = "unpaid";
}
$payment_date   = date("Y-m-d H:i:s");
$payment_method = mysqli_real_escape_string($conn, $_POST['payment_method'] ?? '');
$payment_reference = mysqli_real_escape_string($conn, $_POST['paymentReference'] ?? '');
$remarks        = mysqli_real_escape_string($conn, $_POST['remarks'] ?? '');

$present_days   = (int)$present;
$absent_days    = (int)$absent;
$half_days      = (int)$half_day;
$paid_leaves    = (int)$paid_leave;

$created_by     = (int)($_SESSION['staff_id'] ?? 0);
$updated_by     = 0;



$sql = "
INSERT INTO staff_salary_payment_history (
    staff_id, batch, salary_month, salary_year, total_amount, deductions, net_payable, 
    payment_amount, due_amount, payment_status, payment_date, payment_method, payment_reference, remarks, 
    present_days, absent_days, half_days, paid_leaves, created_by, created_at
) VALUES (
    $staff_id, '$batch', $salary_month, $salary_year, $total_amount, $deductions, $net_payable, 
    $payment_amount, $due_amount, '$payment_status', '$payment_date', '$payment_method', '$payment_reference', '$remarks', 
    $present_days, $absent_days, $half_days, $paid_leaves, $created_by, NOW()
)
";

if ($conn->query($sql) === TRUE) {


echo "<div style='color:green;'>Record inserted successfully.</div>";

// --- Fetch total paid so far from payment history ---
$sql = "SELECT COALESCE(SUM(payment_amount), 0) AS total_paid
        FROM staff_salary_payment_history
        WHERE staff_id = ?
          AND salary_month = ?
          AND salary_year = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("iss", $staff_id, $salary_month, $salary_year);
$stmt->execute();
$result = $stmt->get_result();
$row = $result->fetch_assoc();

$previous_payments = (float)$row['total_paid'];

// Add the current payment to the total
$total_paid_now = $previous_payments;

// Salary to pay — must be defined or fetched from somewhere
$salary_to_pay = (float)($net_payable ?? 0.00);

// Calculate remaining amount
$due_amount = max(0, $salary_to_pay - $total_paid_now);

// Determine payment status
if ($due_amount == 0 && $total_paid_now > 0) {
    $payment_status = "paid";
} elseif ($total_paid_now > 0 && $due_amount > 0) {
    $payment_status = "partially_paid";
} else {
    $payment_status = "unpaid";
}

// Other fields
$payment_type         = $payment_method; // matches ENUM
$transaction_id       = $payment_reference;
$payment_id           = ''; // Optional
$payment_date2        = date("Y-m-d");
$payment_receiver_id  = $created_by;
$remarks2             = $remarks;

// --- Check if salary record exists ---
$check_sql = "SELECT salary_payment_id FROM staff_salary_payments 
              WHERE staff_id = $staff_id 
                AND month = $salary_month 
                AND year = $salary_year 
                AND batch = '$batch'
              LIMIT 1";
$check_result = $conn->query($check_sql);

if ($check_result && $check_result->num_rows > 0) {
    // Update existing record
    $row = $check_result->fetch_assoc();
    $salary_payment_id = (int)$row['salary_payment_id'];

    $update_sql = "
        UPDATE staff_salary_payments SET
            salary_to_pay = $salary_to_pay,
            payment_status = '$payment_status',
            paid_amount = $total_paid_now,
            payment_type = '$payment_type',
            transaction_id = '$transaction_id',
            payment_id = '$payment_id',
            payment_date = '$payment_date2',
            payment_receiver_id = $payment_receiver_id,
            remarks = '$remarks2'
        WHERE salary_payment_id = $salary_payment_id
    ";

    if ($conn->query($update_sql) === TRUE) {
        echo "<div style='color:green;'>Payment summary updated successfully.</div>";
    } else {
        echo "<div style='color:red;'>Error updating payment summary: " . $conn->error . "</div>";
    }

} else {
    // Insert new record
    $sql2 = "
        INSERT INTO staff_salary_payments (
            staff_id, month, year, batch, salary_to_pay,
            payment_status, paid_amount, payment_type, transaction_id, payment_id,
            payment_date, payment_receiver_id, remarks
        ) VALUES (
            $staff_id, $salary_month, $salary_year, '$batch', $salary_to_pay,
            '$payment_status', $total_paid_now, '$payment_type', '$transaction_id', '$payment_id',
            '$payment_date2', $payment_receiver_id, '$remarks2'
        )
    ";

    if ($conn->query($sql2) === TRUE) {
        echo "<div style='color:green;'>Payment summary inserted successfully.</div>";
    } else {
        echo "<div style='color:red;'>Error inserting payment summary: " . $conn->error . "</div>";
    }
}




} else {
echo "Error inserting record: " . $conn->error;
}


}
?>